GILLs_polycubes = (
    [(0,0,0),(1,1,1),(0,0,1),(0,1,1),(0,1,0),(1,0,0),(1,1,0),(1,1,2)],
    [(0,0,0),(0,0,1),(0,1,1),(0,1,0),(1,0,0),(1,1,0),(2,2,-1)],
    )


"""
Redelmeier's algorithm with the untried set organized as a queue.
Prototype implementation for enumerating polyominoes up to size nmax.
The queue is implemented as an array.

In addition, the data are written for graphics display at an html page
"""
from collections import defaultdict

nmax = 10
nmax = 6
PRINT_SOLUTIONS = False

queuelength = nmax * 5 + 10 # polyomino plus neighbors plus safety buffer
Q = [0] * queuelength
occupied_or_adjacent = defaultdict(bool)
count = defaultdict(int)

for x in range(nmax):
    occupied_or_adjacent[x,0,-1] = occupied_or_adjacent[-x,0,0] = True
for x in range(-nmax+1,nmax):
    for y in range(1,nmax):
        occupied_or_adjacent[x,y,-1] = occupied_or_adjacent[x,-y,0] = True
# lower border and starting cell

polyomino = defaultdict(str)
    
def construct(stackbegin, stackend, n):
    """Current polyomino has n cells.
    UNTRIED points are stored in Q[stackbegin] ... Q[stackend-1]."""
    count[n] += 1
    store(polyomino)
    if PRINT_SOLUTIONS:
        polyomino[0,0,0]="S" # mark the start position
        print_grid(polyomino, text = f" {n=}, number {count[n]}")
        print()
    if n>=nmax:
        return
    for i in range(stackbegin, stackend):
        #print(f"{n=} {i=} {stackbegin}:{stackend} {Q[stackbegin:stackend]}",)
        #print_grid(occupied_or_adjacent)
        x,y,z = Q[i]
        # include the cell (x,y):
        polyomino[x,y,z] = "X" # needed only for printing
        #occupied_or_adjacent[x,y] = "S" # helpful for debugging
        new_neighbors = [nbr for nbr in ((x-1,y,z),(x,y-1,z),
                                         (x+1,y,z),(x,y+1,z),
                                         (x,y,z-1),(x,y,z+1))
                         if not occupied_or_adjacent[nbr]]
        for k,nbr in enumerate(new_neighbors):
            occupied_or_adjacent[nbr]=True
            Q[stackend+k] = nbr
            
        # recursive call:
        construct(i+1, stackend+len(new_neighbors), n+1)
        
        for nbr in new_neighbors:
            occupied_or_adjacent[nbr]=False # undo the mark; nbr becomes "unseen"
        polyomino[x,y,z] = " " # needed only for printing
        #occupied_or_adjacent[x,y] = True # reset the debugging marker

Polyominoes = defaultdict(set)


def unique_representative(po, with_mirror=False):
    xmin = min(x for x,y,z in po)
    ymin = min(y for x,y,z in po)
    zmin = min(z for x,y,z in po)
    po = [(x-xmin,y-ymin,z-zmin) for x,y,z in po] # normalize position
    xmax = max(x for x,y,z in po)
    ymax = max(y for x,y,z in po)
    zmax = max(z for x,y,z in po)
    dims = xmax,ymax,zmax
    return tuple(
    min(sorted((s1*X[p1] + (1-s1)//2*dims[p1],
                s2*X[p2] + (1-s2)//2*dims[p2],
                s3*X[p3] + (1-s3)//2*dims[p3])
               for X in po)
        for (p1,p2,p3),s in [((0,1,2),+1),((1,2,0),+1),((2,0,1),+1),
                             ((0,2,1),-1),((1,0,2),-1),((2,1,0),-1),]
        if dims[p1]>=dims[p2]>=dims[p3]
        for s1 in (-1,+1)
        for s2 in (-1,+1)
        for s3 in (-1,+1)
        if with_mirror or s*s1*s2*s3==+1 # positive orientation
        ))
def mirror(po):
    return [(y,x,z) for x,y,z in po]

def lift_to_3d(v,dim,z):
    "lift 2d vector v to 3 dimensions by inserting coordinate z"
    x = list(v)
    x[dim:dim] = [z]
    return tuple(x)



plane_polyominoes = dict()
plane_polyominoes[3,'L']=((0, 0), (0, 1), (1, 0))
plane_polyominoes[3,'I']=((0, 0), (1, 0), (2, 0))
plane_polyominoes[4,'o']=((0, 0), (0, 1), (1, 0), (1, 1))
plane_polyominoes[4,'l']=((0, 0), (0, 1), (1, 0), (2, 0))
plane_polyominoes[4,'t']=((0, 0), (1, 0), (1, 1), (2, 0))
plane_polyominoes[4,'n']=((0, 0), (1, 0), (1, 1), (2, 1))
plane_polyominoes[4,'i']=((0, 0), (1, 0), (2, 0), (3, 0))
plane_polyominoes[5,'V']=((0, 0), (0, 1), (0, 2), (1, 0), (2, 0))
plane_polyominoes[5,'P']=((0, 0), (0, 1), (1, 0), (1, 1), (2, 0))
plane_polyominoes[5,'U']=((0, 0), (0, 1), (1, 0), (2, 0), (2, 1))
plane_polyominoes[5,'L']=((0, 0), (0, 1), (1, 0), (2, 0), (3, 0))
plane_polyominoes[5,'T']=((0, 0), (1, 0), (1, 1), (1, 2), (2, 0))
plane_polyominoes[5,'F']=((0, 0), (1, 0), (1, 1), (1, 2), (2, 1))
plane_polyominoes[5,'Z']=((0, 0), (1, 0), (1, 1), (1, 2), (2, 2))
plane_polyominoes[5,'Y']=((0, 0), (1, 0), (1, 1), (2, 0), (3, 0))
plane_polyominoes[5,'W']=((0, 0), (1, 0), (1, 1), (2, 1), (2, 2))
plane_polyominoes[5,'N']=((0, 0), (1, 0), (2, 0), (2, 1), (3, 1))
plane_polyominoes[5,'I']=((0, 0), (1, 0), (2, 0), (3, 0), (4, 0))
plane_polyominoes[5,'X']=((0, 1), (1, 0), (1, 1), (1, 2), (2, 1))


polyo_name_list = {
    unique_representative([lift_to_3d(v,2,0) for v in po_2d]) : letter 
    for (_,letter),po_2d in plane_polyominoes.items()}

polyo_name_list[((0,0,0), (1,0,0), (1,1,0), (1,1,1), (2,1,1), (2,2,1))] = "right screw"
polyo_name_list[((0,0,0), (0,1,0), (1,1,0), (1,1,1), (1,2,1), (2,2,1))] = "left screw"
polyo_name_list[((0,0,0), (0,0,1), (1,0,0), (1,1,0), (2,0,0), (2,0,1))] = 'throne'
polyo_name_list[((0,0,0), (0,0,1), (0,1,0), (0,1,1), (1,0,0), (1,0,1))] = 'bench'
polyo_name_list[((0,0,0), (1,0,0), (1,0,1), (1,1,1), (2,1,1)) ] = "left screw"
polyo_name_list[((0,0,0), (1,0,0), (1,1,0), (1,1,1), (2,1,1)) ] = "right screw"
polyo_name_list[((0,0,0), (0,1,0), (0,2,0), (1,0,0), (1,2,0), (2,0,0)) ] = 'J'
polyo_name_list[((0,0,0), (0,1,0), (0,2,0), (1,1,0), (2,0,0), (2,1,0)) ] = 'h'
polyo_name_list[((0,0,0), (1,0,0), (1,1,0), (2,1,0), (2,2,0), (3,2,0)) ] = 'snake' # zigzag,
polyo_name_list[((0,0,0), (0,1,0), (0,2,0), (1,0,0), (1,1,0), (2,0,0)) ] = 'stairs'
polyo_name_list[((0,0,0), (0,1,0), (1,0,0), (2,0,0), (2,1,0), (3,0,0)) ] = 'F'
polyo_name_list[((0,1,0), (1,0,0), (1,1,0), (1,2,0), (2,1,0), (3,1,0)) ] = 'Latin cross'
#polyo_name_list[(( ] = 'xxx'
#polyo_name_list[(( ] = 'xxx'
#polyo_name_list[(( ] = 'xxx'





def store(GRID):
    po = set(x for x,letter in GRID.items() if letter != " ")
    n = len(po)
    if n==0:
        return
    Polyominoes[n].add(unique_representative(po))

def print_3d_cubes(po):
    vertices,edges,squares,directions = make_surface_3d_cubes(po)
    xmin = min(x for x,y,z in po)
    ymin = min(y for x,y,z in po)
    zmin = min(z for x,y,z in po)
    xmax = max(x for x,y,z in po)
    ymax = max(y for x,y,z in po)
    zmax = max(z for x,y,z in po)
    shiftx = (xmin-xmax)/2
    shifty = (ymin-ymax)/2
    shiftz = (zmin-zmax)/2
    
    #vertices = sorted(set(v for s in edges["Thick"]+edges["thin"] for v in s))
    vertexnumber = {v:i for i,v in enumerate(vertices)}
    out='"vertex":[' + ",".join("[%g,%g,%g]" % (v[0]+shiftx,v[1]+shifty,v[2]+shiftz)
                                for v in vertices) + "],\n"
    out+='"edge":[' + ",".join("[%d,%d]" % tuple(vertexnumber[v] for v in e)
                               for e in edges["Thick"]+edges["thin"]
                               ) + "],\n"
    out+=f'"numthickedges":{len(edges["Thick"])},\n'
    out+='"face":[' + ",".join("["
                    +",".join(str(vertexnumber[v]) for v in p) + "]"
                               for p in squares)  + "],\n"
    out+='"facedirection":[' + ",".join(str(d) for d in directions)  + "]"
    return(out)
    
                               
def make_surface_3d_cubes(po):
    "make a 3d model with vertices, edges, and faces"
    squares = defaultdict(list)
    All_edges = set() # union of all edge sets of all cubes
    for p in po:
        for dim in 0,1,2:
            for offset in -1,+1:
                x = list(p)
                height = p[dim]+offset/2
                x[dim]+=offset
                if tuple(x) not in po:
                    "found a boundary face"
                    location = tuple(v for i,v in enumerate(p) if i!=dim)
                    squares[dim,height,offset].append(location)
                del x[dim]
                a,b = x
                for (u1,v1),(u2,v2) in (((+1,+1),(+1,-1)),((+1,-1),(-1,-1)),((-1,-1),(-1,+1)),((-1,+1),(+1,+1))):
                    ed = (a+u1/2,b+v1/2),(a+u2/2,b+v2/2)
                    All_edges.add(frozenset(lift_to_3d(c,dim,height) for c in ed)) # a little bit overkill
    All_vertices = set(c for ed in All_edges for c in ed) # union of all vertices of all cubes
    polygons = []
    for (dim,height,dir),f in squares.items():
        # "dir" means: upward and downward faces in the same plane are distinguished
        edges = []
        all_edges = set()
        for a,b in f:
            for dim2 in 0,1:
                for offset in -1,+1:
                    x = [a,b]
                    x[dim2]+=offset
                    if dim2==0:
                        u,v=(a+offset/2,b-0.5), (a+offset/2,b+0.5)
                    else:
                        u,v=(a+0.5,b+offset/2), (a-0.5,b+offset/2)
                    if offset<0:
                        u,v = v,u
                    # edge is now properly oriented:
                    # inside is on the left side.
                    if tuple(x) not in f:
                        "found a boundary edge"
                        edges.append((u,v))
        # now chain the edges together
        #print (f"{((dim,height,dir),f)=} {edges=}")
        out = defaultdict(list)
        for u,v in edges:
            out[u].append(v)
        used_edges = set()
        chains = []
        for u,v in edges:
            if (u,v) in used_edges:
                continue
            path = []
            while (u,v) not in used_edges:
                used_edges.add((u,v))
                path.append((u,v))
                ww = out[v]
                if len(ww)==1:
                    u,v = v,ww[0]
                else:
                    assert len(ww)==2
                    dx, dy = v[0]-u[0], v[1]-u[1] # incoming direction
                    ddx, ddy = -dy, dx # ccw by 90°
                    new = v[0]+ddx,v[1]+ddy
                    assert new in ww
                    u,v = v,new
            # simplify straight edges
            assert len(path)>=4
            assert all(u2==v1 for (u1,u2),(v1,v2) in zip(path,path[1:]))
            assert all(u2==v1 for (u1,u2),(v1,v2) in [(path[-1],path[0])])
            steps = [(u2[0]-u1[0],u2[1]-u1[1]) for (u1,u2) in path]
            corners = [u for i,(u,v) in enumerate(path) if steps[i]!=steps[i-1]]

            face = [lift_to_3d(c,dim,height) for c in corners] # lift to 3 dimensions:
            polygons.append(face)
            #print (f"{((dim,height,dir),f)=} {path=}")
            #print (f"{((dim,height,dir),f)=} {corners=}")
    vertices = set(v for p in polygons for v in p)
    edges = set(frozenset([p[i-1],v]) for p in polygons for i,v in enumerate(p))

    # consolidate multiple edges along the same line:
    # find the maximal intervals along each line.
    # This amounts to forming a union of intervals
    lines = defaultdict(list)
    for u,v in edges:
        dim, start, end, rest = compute_start_end(u,v)
        lines[dim,rest]+=[((start,+1,"Thick")),((end,-1,"Thick"))]
    for u,v in All_edges:
        dim, start, end, rest = compute_start_end(u,v)
        lines[dim,rest]+=[((start,+1,"thin")),((end,-1,"thin"))]
        
    segments = {"Thick":[], "thin": []}
    for (dim,rest),zlist in lines.items():
        current_status = ""
        num_covered = {"Thick":0, "thin": 0}
        prev = None
        for (z,sign,kind) in sorted(zlist):
            if z!=prev:
                # all signs at "prev" have been accumulated
                if num_covered["Thick"]>0:
                    new_status = "Thick" # Thick takes precedence over thin
                elif num_covered["thin"]>0:
                    new_status = "thin"
                else: # num_covered==0
                    new_status = ""
                if new_status != current_status:
                    if current_status:
                        segments[current_status].append((lift_to_3d(rest,dim,start),
                                                         lift_to_3d(rest,dim,prev)))
                    start = prev
                current_status = new_status
            prev = z
            num_covered[kind] += sign
        #if num_covered==0 and current_status: # (must be active)
        segments[current_status].append((lift_to_3d(rest,dim,start),
                                         lift_to_3d(rest,dim,prev)))
            
    
    # polygons cannot be used, because of non-convexity.
    # use the squares instead. (polygons that are star-shaped
    # from the first vertex would also be ok.)

    polygons = [
        [lift_to_3d((a+0.5*offsetx,b+0.5*offsety),dim,height)
         for offsetx, offsety in ((-1,-sign),(1,-sign),(1,sign),(-1,sign))]
        # use proper orientation, depending on dir.
        # Currently the surface is visible from outside in, but not
        # from inside out.
        for (dim,height,dir),f in squares.items()
        for sign in [(-1)**dim * dir]
        for a,b in f ]
    directions = [ (dim+1)*dir
                   for (dim,height,dir),f in squares.items()
                   for a,b in f ]    
    return All_vertices,segments,polygons,directions
                    
def compute_start_end(u,v):        
    for dim in 0,1,2:
        if u[dim]!=v[dim]:
            break
    start = min(u[dim],v[dim])
    end   = max(u[dim],v[dim])
    rest = list(u)
    del rest[dim]
    return dim, start, end, tuple(rest)

print (print_3d_cubes([(1,2,3)]))
print (print_3d_cubes([(1,2,3),(0,1,3)]))
print (print_3d_cubes([(1,2,3),(1,3,3)]))
            
        
    
def print_cube(po):    
    xmax = max(x for x,y,z in po)
    ymax = max(y for x,y,z in po)
    zmax = max(z for x,y,z in po)
    lines = [""]*(ymax+1)
    for z in range(zmax+1):
        text = f"  Schicht {z}: "
        if z:
            text = ","+text
        lines[0] += text
        for y in range(1,1+ymax):
            lines[y] += " "*len(text)
        pattern = [[" " for _ in range(1+xmax)] for _ in range(1+ymax)]
        for (x,y,z0) in po:
            if z==z0:
                pattern[y][x]="X"
        for y in range(1+ymax):
            lines[y] += "["+"".join(pattern[ymax-y]) + "]"
    print("\n".join(lines))
    print()
       
        
def print_grid(GRID, text=""):
    # print the pattern represented in the dictionary GRID
    xmin = min(x for x,y in GRID.keys())
    xmax = max(x for x,y in GRID.keys())
    ymin = min(y for x,y in GRID.keys())
    ymax = max(y for x,y in GRID.keys())
    pattern = [["." for _ in range(1+xmax-xmin)] for _ in range(1+ymax-ymin)]
    for (x,y),letter in GRID.items():
        if letter is False:
            letter = " "
        elif letter is True:
            letter = "X"
        pattern[ymax-y][x-xmin]=letter
    print("\n".join("".join(l) for l in pattern)+text + f",  {xmin}<=x<={xmax}, {ymin}<=y<={ymax}")
       
Q[0] = (0,0,0) # The starting square (0,0) is put on the queue
construct(0, 1, 0) # start the enumeration
for i,val in sorted(count.items()):
    print(f"{i:2} {val:6}") # results
for i,val in sorted(Polyominoes.items()):
    print(f"{i:2} {len(val):6}")

Polyominoes = {n:sorted(ppp) for n,ppp in Polyominoes.items()} # turn to lists
print(Polyominoes[2])
print(Polyominoes[3])

Classification = defaultdict(list)
renumbering = defaultdict(list)
for n,ppp in Polyominoes.items():
    for i,po in enumerate(ppp):
        name = ""
        if all(z==0 for x,y,z in po):
            clas = "plane"
            if po in polyo_name_list:
                if len(polyo_name_list[po])==1:
                    clas += ", type "+polyo_name_list[po]
                else:
                    name = ", '"+polyo_name_list[po]+"'"
            renumbering[n].append((("0plane",),i))
        else:
            po2 = unique_representative(mirror(po))
            if po2==po:
                clas = "symmetric"
                renumbering[n].append((("1symmetric",),i))
            else:
                partner = ppp.index(po2)
                clas = f"mirror of #{partner+1}"
                canonical = unique_representative(po, with_mirror=True)
                if po==canonical:
                    clas += ", v.A"
                    renumbering[n].append((("2pair",i,0),i))
                else:
                    clas += ", v.B"            
                    renumbering[n].append((("2pair",partner,1),i))
            if po in polyo_name_list:
                name = ", '"+polyo_name_list[po]+"'"
        Classification[n].append((clas,name))

        
for i,(po,(clas,name)) in enumerate(zip(Polyominoes[n],Classification[n])):
    print(f"Pentawürfel #{i+1}{name} ({clas}):", po)
    #print(f"Hexawürfel #{i+1}:", po)
    print_cube(po)

Polyominoes[0] = GILLs_polycubes
Classification[0] =[("","") for i in range(len(GILLs_polycubes))]
renumbering[0]=[("Gill",i) for i in range(len(GILLs_polycubes))]

    
fname = "js/Polycubes-data.js"
with open(fname,'w') as aus:
  aus.write("""/**
 * Data made by Günter Rote with a python program
 */
 
POLYHEDRA = {

""")
  for n,nam1,nam2 in [(0,"Gill","Gill"),
          (3,"tri","Tri"),(4,"tetra","Tetra"),
                        (5,"penta","Penta"),(6,"hexa","Hexa")]:
      cat = nam2+"cubes"
      for i,(key,num) in enumerate(sorted(renumbering[n])):
        po = Polyominoes[n][num]
        clas,name = Classification[n][num]
        if key[0]=="2pair":
            if key[2]==0: # first element of pair
                clas = f"mirror of #{i+2}, 1st"
            else:
                clas = f"mirror of #{i}, 2nd"
        name1 = f"{nam2}{i+1}"
        name2 = f"{nam1}#{i+1}{name} ({clas})"
        data=print_3d_cubes(po)
        aus.write(f'''
{name1} : {{
"name":"{name2}",
"category":["{cat}"],
"cell":[{', '.join('['+','.join(str(x) for x in pu)+']' for pu in po)}],
{data}}},
''')
  aus.write("};\n")
print("file",fname,"written.")
